/**
 * Google Sheets Service
 * Handles communication with Google Apps Script to update subscriber sheet
 */
class GoogleSheetsService {
    static webAppUrl = process.env.GOOGLE_APPS_SCRIPT_URL;
    static isEnabled = false;

    /**
     * Initialize the service
     */
    static async initialize() {
        console.log('📊 Initializing Google Sheets service...');

        if (!GoogleSheetsService.webAppUrl) {
            console.log('   ⚠️  GOOGLE_APPS_SCRIPT_URL not set in environment');
            console.log('   Google Sheets integration disabled');
            return false;
        }

        // Test the connection
        try {
            const response = await fetch(GoogleSheetsService.webAppUrl, {
                method: 'GET',
                headers: {
                    'Content-Type': 'application/json'
                }
            });

            if (response.ok) {
                GoogleSheetsService.isEnabled = true;
                console.log('   ✓ Google Sheets API connected');
                return true;
            } else {
                console.log('   ✗ Google Sheets API not responding');
                return false;
            }
        } catch (error) {
            console.log('   ✗ Failed to connect to Google Sheets API');
            console.log('   Error:', error.message);
            return false;
        }
    }

    /**
     * Add a subscriber to Google Sheets
     * @param {Object} subscriber - The subscriber data
     */
    static async addSubscriber(subscriber) {
        if (!GoogleSheetsService.isEnabled) {
            console.log('⚠️  Google Sheets disabled, skipping...');
            return { success: false, message: 'Service not enabled' };
        }

        try {
            console.log('📊 Adding subscriber to Google Sheet...');
            console.log('   Email:', subscriber.email);

            const response = await fetch(GoogleSheetsService.webAppUrl, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    id: subscriber.id,
                    firstName: subscriber.firstName,
                    lastName: subscriber.lastName,
                    email: subscriber.email,
                    verified: subscriber.verified,
                    subscribedAt: subscriber.subscribedAt,
                    verifiedAt: subscriber.verifiedAt || '',
                    source: subscriber.source || 'website'
                })
            });

            const result = await response.json();

            if (result.success) {
                console.log('   ✓ Successfully added to Google Sheet');
                return { success: true };
            } else {
                console.log('   ✗ Failed to add to Google Sheet');
                console.log('   Error:', result.error);
                return { success: false, error: result.error };
            }
        } catch (error) {
            console.error('❌ Error adding to Google Sheet:', error.message);
            return { success: false, error: error.message };
        }
    }

    /**
     * Update subscriber verification status
     * Note: Apps Script doesn't easily support updates, so we just add the verified row
     * @param {Object} subscriber - The verified subscriber data
     */
    static async updateSubscriberVerification(subscriber) {
        // For Apps Script, we just add a new row when verified
        return await GoogleSheetsService.addSubscriber(subscriber);
    }
}

export default GoogleSheetsService;