import express from 'express';
import cors from 'cors';
import SubscriptionService from './services/subscriptionService.js';
import EmailService from './services/emailService.js';
import ArticleService from './services/articleService.js';
import GoogleSheetsService from './services/GoogleSheetsService.js';

import { fileURLToPath } from 'url';
import { dirname } from 'path';

const __filename = fileURLToPath(import.meta.url);
const __dirname = dirname(__filename);

/**
 * Express service for handling HTTP requests
 */
class ExpressService {
  static port = process.env.PORT || 5001;
  static nodeEnvironment = process.env.NODE_ENV || 'development';
  static mainSiteUrl = 'https://tlchomes.co.ke';
  static dirname = dirname(fileURLToPath(import.meta.url));

  #app;
  #subscriberRouter;

  constructor() {
    this.#app = express();
    this.#subscriberRouter = express.Router();
  }

  // In your ExpressService class, update the #initialize() method
  // Replace the existing CORS configuration (around line 40-55) with this:

  #initialize() {
    // CORS configuration - UPDATED TO FIX LOCALHOST ACCESS
    this.#app.use(cors({
      origin: (origin, callback) => {
        const allowedOrigins = process.env.NODE_ENV === 'production'
          ? [
            'https://tlchomes.co.ke',
            'https://www.tlchomes.co.ke',
            'https://newsletter.tlchomes.co.ke'
          ]
          : [
            'http://localhost:5173',
            'http://localhost:5001',
            'http://localhost:3000',
            'http://127.0.0.1:5173'
          ];

        // Allow requests with no origin (like mobile apps or Postman)
        if (!origin) return callback(null, true);

        if (allowedOrigins.includes(origin)) {
          callback(null, true);
        } else {
          console.log(`❌ CORS blocked origin: ${origin}`);
          callback(new Error('Not allowed by CORS'));
        }
      },
      methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
      allowedHeaders: ['Content-Type', 'Authorization', 'Accept'],
      credentials: true,
      optionsSuccessStatus: 200
    }));

    // Handle preflight requests explicitly
    this.#app.options('*', cors());

    // Body parsers
    this.#app.use(express.json());
    this.#app.use(express.urlencoded({ extended: true }));

    // Trust proxy and security headers
    this.#app.set('trust proxy', 1);
    this.#app.use((req, res, next) => {
      res.setHeader('X-Content-Type-Options', 'nosniff');
      res.setHeader('X-Frame-Options', 'DENY');
      res.setHeader('X-XSS-Protection', '1; mode=block');
      if (process.env.NODE_ENV === 'production') {
        res.setHeader('Strict-Transport-Security', 'max-age=31536000; includeSubDomains');
      }
      next();
    });

    // Mount subscriber router with /backend prefix
    this.#app.use('/backend/api/subscriber', this.#subscriberRouter);

    // Also mount without prefix for backward compatibility
    this.#app.use('/api/subscriber', this.#subscriberRouter);

    // Logging middleware
    this.#loggingMiddleware();
  }
  /**
   * Logging middleware for monitoring
   */
  #loggingMiddleware() {
    this.#app.use(async (req, res, next) => {
      console.log(`[${new Date().toISOString()}] ${req.method} ${req.path}`);
      next();
    });
  }

  /**
   * POST /api/subscriber/add
   * Add a new subscriber
   */
  #addSubscriberRoute() {
    this.#subscriberRouter.post('/add', async (req, res) => {
      console.log('🔵 POST /api/subscriber/add called');
      console.log('Request body:', req.body);

      try {
        const { email, firstName, lastName, source } = req.body;

        // Validate required fields
        if (!email || !firstName || !lastName) {
          console.log('❌ Validation failed: missing fields');
          return res.status(400).json({
            success: false,
            message: 'Email, first name, and last name are required'
          });
        }

        // Validate email format
        const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        if (!emailRegex.test(email)) {
          console.log('❌ Validation failed: invalid email format');
          return res.status(400).json({
            success: false,
            message: 'Invalid email format'
          });
        }

        console.log('✓ Validation passed, calling SubscriptionService...');

        const result = await SubscriptionService.subscribe(
          email,
          firstName,
          lastName,
          source || 'website'
        );

        console.log('✓ SubscriptionService returned:', result);

        // Handle already subscribed case
        if (!result.success && result.message?.includes('already subscribed')) {
          return res.status(400).json(result);
        }

        return res.status(200).json({
          success: true,
          message: result.message || 'Subscription successful!',
          emailSent: result.emailSent !== false
        });

      } catch (error) {
        console.error('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        console.error('❌ CRITICAL ERROR in add subscriber route:');
        console.error('Error message:', error.message);
        console.error('Error name:', error.name);
        console.error('Error stack:', error.stack);
        console.error('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');

        try {
          return res.status(500).json({
            success: false,
            message: 'An error occurred. Please try again later.',
            error: ExpressService.nodeEnvironment === 'development' ? error.message : undefined
          });
        } catch (sendError) {
          console.error('❌ Failed to send error response:', sendError);
        }
      }
    });
  }

  /**
   * GET /api/subscriber/verify
   * Verify subscriber email
   */
  #verifySubscriberRoute() {
    this.#subscriberRouter.get('/verify', async (req, res, next) => {
      try {
        const { id, email } = req.query;

        if (!id || !email) {
          return res.status(400).send(`
            <!DOCTYPE html>
            <html lang="en">
              <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>Invalid Link</title>
                <style>
                  body { 
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif;
                    background: #f5f5f5;
                    margin: 0;
                    padding: 40px 20px;
                  }
                  .container { 
                    max-width: 600px;
                    margin: 0 auto;
                    background: white;
                    padding: 60px 40px;
                    text-align: center;
                  }
                  h1 { 
                    font-size: 24px;
                    font-weight: 400;
                    color: #202124;
                    margin: 0 0 20px 0;
                  }
                  p { 
                    font-size: 16px;
                    color: #5f6368;
                    line-height: 1.5;
                    margin: 0 0 32px 0;
                  }
                  a {
                    display: inline-block;
                    padding: 12px 32px;
                    background: #000;
                    color: white;
                    text-decoration: none;
                    border-radius: 4px;
                    font-size: 14px;
                    font-weight: 500;
                  }
                </style>
              </head>
              <body>
                <div class="container">
                  <h1>Invalid verification link</h1>
                  <p>The verification link is invalid or incomplete.</p>
                  <a href="${ExpressService.mainSiteUrl}">Return to TLC Homes</a>
                </div>
              </body>
            </html>
          `);
        }

        const result = await SubscriptionService.verifySubscriber(id, email);

        if (!result.success) {
          return res.status(400).send(`
            <!DOCTYPE html>
            <html lang="en">
              <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>Verification Failed</title>
                <style>
                  body { 
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif;
                    background: #f5f5f5;
                    margin: 0;
                    padding: 40px 20px;
                  }
                  .container { 
                    max-width: 600px;
                    margin: 0 auto;
                    background: white;
                    padding: 60px 40px;
                    text-align: center;
                  }
                  h1 { 
                    font-size: 24px;
                    font-weight: 400;
                    color: #202124;
                    margin: 0 0 20px 0;
                  }
                  p { 
                    font-size: 16px;
                    color: #5f6368;
                    line-height: 1.5;
                    margin: 0 0 32px 0;
                  }
                  a {
                    display: inline-block;
                    padding: 12px 32px;
                    background: #000;
                    color: white;
                    text-decoration: none;
                    border-radius: 4px;
                    font-size: 14px;
                    font-weight: 500;
                  }
                </style>
              </head>
              <body>
                <div class="container">
                  <h1>Verification failed</h1>
                  <p>${result.message}</p>
                  <a href="${ExpressService.mainSiteUrl}">Return to TLC Homes</a>
                </div>
              </body>
            </html>
          `);
        }

        return res.status(200).send(`
          <!DOCTYPE html>
          <html lang="en">
            <head>
              <meta charset="UTF-8">
              <meta name="viewport" content="width=device-width, initial-scale=1.0">
              <title>Email Verified</title>
              <style>
                body { 
                  font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif;
                  background: #f5f5f5;
                  margin: 0;
                  padding: 40px 20px;
                }
                .container { 
                  max-width: 600px;
                  margin: 0 auto;
                  background: white;
                  padding: 60px 40px;
                  text-align: center;
                }
                h1 { 
                  font-size: 24px;
                  font-weight: 400;
                  color: #202124;
                  margin: 0 0 20px 0;
                }
                p { 
                  font-size: 16px;
                  color: #5f6368;
                  line-height: 1.5;
                  margin: 0 0 32px 0;
                }
                a {
                  display: inline-block;
                  padding: 12px 32px;
                  background: #000;
                  color: white;
                  text-decoration: none;
                  border-radius: 4px;
                  font-size: 14px;
                  font-weight: 500;
                }
              </style>
            </head>
            <body>
              <div class="container">
                <h1>Email verified</h1>
                <p>Welcome, ${result.subscriber.firstName}!</p>
                <p>${result.message}</p>
                <a href="${ExpressService.mainSiteUrl}">Visit TLC Homes Services</a>
              </div>
            </body>
          </html>
        `);
      } catch (error) {
        console.error('❌ Error in verify subscriber route:', error);
        return res.status(500).send(`
          <!DOCTYPE html>
          <html lang="en">
            <head>
              <meta charset="UTF-8">
              <meta name="viewport" content="width=device-width, initial-scale=1.0">
              <title>Error</title>
              <style>
                body { 
                  font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif;
                  background: #f5f5f5;
                  margin: 0;
                  padding: 40px 20px;
                }
                .container { 
                  max-width: 600px;
                  margin: 0 auto;
                  background: white;
                  padding: 60px 40px;
                  text-align: center;
                }
                h1 { 
                  font-size: 24px;
                  font-weight: 400;
                  color: #202124;
                  margin: 0 0 20px 0;
                }
                p { 
                  font-size: 16px;
                  color: #5f6368;
                  line-height: 1.5;
                  margin: 0 0 32px 0;
                }
                a {
                  display: inline-block;
                  padding: 12px 32px;
                  background: #000;
                  color: white;
                  text-decoration: none;
                  border-radius: 4px;
                  font-size: 14px;
                  font-weight: 500;
                }
              </style>
            </head>
            <body>
              <div class="container">
                <h1>Server error</h1>
                <p>An unexpected error occurred. Please try again later.</p>
                <a href="${ExpressService.mainSiteUrl}">Return home</a>
              </div>
            </body>
          </html>
        `);
      }
    });
  }

  /**
   * GET /api/subscriber/unsubscribe
   * Unsubscribe from newsletter
   */
  #unsubscribeRoute() {
    this.#subscriberRouter.get('/unsubscribe', async (req, res, next) => {
      try {
        const { id, email } = req.query;

        if (!id || !email) {
          return res.status(400).send(`
            <!DOCTYPE html>
            <html lang="en">
              <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>Invalid Link</title>
                <style>
                  body { 
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif;
                    background: #f5f5f5;
                    margin: 0;
                    padding: 40px 20px;
                  }
                  .container { 
                    max-width: 600px;
                    margin: 0 auto;
                    background: white;
                    padding: 60px 40px;
                    text-align: center;
                  }
                  h1 { 
                    font-size: 24px;
                    font-weight: 400;
                    color: #202124;
                    margin: 0 0 20px 0;
                  }
                  p { 
                    font-size: 16px;
                    color: #5f6368;
                    line-height: 1.5;
                    margin: 0 0 32px 0;
                  }
                  a {
                    display: inline-block;
                    padding: 12px 32px;
                    background: #000;
                    color: white;
                    text-decoration: none;
                    border-radius: 4px;
                    font-size: 14px;
                    font-weight: 500;
                  }
                </style>
              </head>
              <body>
                <div class="container">
                  <h1>Invalid unsubscribe link</h1>
                  <p>The unsubscribe link is invalid or incomplete.</p>
                  <a href="${ExpressService.mainSiteUrl}">Return to TLC Homes Services</a>
                </div>
              </body>
            </html>
          `);
        }

        const result = await SubscriptionService.unsubscribe(id, email);

        if (!result.success) {
          return res.status(400).send(`
            <!DOCTYPE html>
            <html lang="en">
              <head>
                <meta charset="UTF-8">
                <meta name="viewport" content="width=device-width, initial-scale=1.0">
                <title>Unsubscribe Failed</title>
                <style>
                  body { 
                    font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif;
                    background: #f5f5f5;
                    margin: 0;
                    padding: 40px 20px;
                  }
                  .container { 
                    max-width: 600px;
                    margin: 0 auto;
                    background: white;
                    padding: 60px 40px;
                    text-align: center;
                  }
                  h1 { 
                    font-size: 24px;
                    font-weight: 400;
                    color: #202124;
                    margin: 0 0 20px 0;
                  }
                  p { 
                    font-size: 16px;
                    color: #5f6368;
                    line-height: 1.5;
                    margin: 0 0 32px 0;
                  }
                  a {
                    display: inline-block;
                    padding: 12px 32px;
                    background: #000;
                    color: white;
                    text-decoration: none;
                    border-radius: 4px;
                    font-size: 14px;
                    font-weight: 500;
                  }
                </style>
              </head>
              <body>
                <div class="container">
                  <h1>Unsubscribe failed</h1>
                  <p>${result.message}</p>
                  <a href="${ExpressService.mainSiteUrl}">Return to TLC Homes Services</a>
                </div>
              </body>
            </html>
          `);
        }

        return res.status(200).send(`
          <!DOCTYPE html>
          <html lang="en">
            <head>
              <meta charset="UTF-8">
              <meta name="viewport" content="width=device-width, initial-scale=1.0">
              <title>Unsubscribed</title>
              <style>
                body { 
                  font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif;
                  background: #f5f5f5;
                  margin: 0;
                  padding: 40px 20px;
                }
                .container { 
                  max-width: 600px;
                  margin: 0 auto;
                  background: white;
                  padding: 60px 40px;
                  text-align: center;
                }
                h1 { 
                  font-size: 24px;
                  font-weight: 400;
                  color: #202124;
                  margin: 0 0 20px 0;
                }
                p { 
                  font-size: 16px;
                  color: #5f6368;
                  line-height: 1.5;
                  margin: 0 0 8px 0;
                }
                .footer {
                  margin-top: 32px;
                  padding-top: 32px;
                  border-top: 1px solid #dadce0;
                }
                a {
                  display: inline-block;
                  padding: 12px 32px;
                  background: #000;
                  color: white;
                  text-decoration: none;
                  border-radius: 4px;
                  font-size: 14px;
                  font-weight: 500;
                }
              </style>
            </head>
            <body>
              <div class="container">
                <h1>Unsubscribed successfully</h1>
                <p>Hi ${result.subscriber.firstName},</p>
                <p>${result.message}</p>
                <p>You can resubscribe anytime on our website.</p>
                <div class="footer">
                  <a href="${ExpressService.mainSiteUrl}">Visit TLC Homes Services</a>
                </div>
              </div>
            </body>
          </html>
        `);
      } catch (error) {
        console.error('❌ Error in unsubscribe route:', error);
        return res.status(500).send(`
          <!DOCTYPE html>
          <html lang="en">
            <head>
              <meta charset="UTF-8">
              <meta name="viewport" content="width=device-width, initial-scale=1.0">
              <title>Error</title>
              <style>
                body { 
                  font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, Arial, sans-serif;
                  background: #f5f5f5;
                  margin: 0;
                  padding: 40px 20px;
                }
                .container { 
                  max-width: 600px;
                  margin: 0 auto;
                  background: white;
                  padding: 60px 40px;
                  text-align: center;
                }
                h1 { 
                  font-size: 24px;
                  font-weight: 400;
                  color: #202124;
                  margin: 0 0 20px 0;
                }
                p { 
                  font-size: 16px;
                  color: #5f6368;
                  line-height: 1.5;
                  margin: 0 0 32px 0;
                }
                a {
                  display: inline-block;
                  padding: 12px 32px;
                  background: #000;
                  color: white;
                  text-decoration: none;
                  border-radius: 4px;
                  font-size: 14px;
                  font-weight: 500;
                }
              </style>
            </head>
            <body>
              <div class="container">
                <h1>Server error</h1>
                <p>An unexpected error occurred. Please try again later.</p>
                <a href="${ExpressService.mainSiteUrl}">Return home</a>
              </div>
            </body>
          </html>
        `);
      }
    });
  }

  /**
 * POST /api/subscriber/notify
 * Send blog notification to all subscribers (ADMIN ONLY)
 */

  #notifySubscribersRoute() {
    this.#subscriberRouter.post('/notify', async (req, res, next) => {
      console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
      console.log('🔵 POST /api/subscriber/notify');
      console.log('Headers:', req.headers['content-type']);
      console.log('Body keys:', Object.keys(req.body));

      try {
        const {
          password,
          title,
          excerpt,
          link,
          imageUrl,
          author,
          publishedDate,
          category,
          tags
        } = req.body;

        // Debug password
        console.log('Password check:');
        console.log('  Received:', password ? `✓ (length: ${password.length})` : '✗ MISSING');
        console.log('  Expected:', process.env.ADMIN_PASSWORD ? `✓ (length: ${process.env.ADMIN_PASSWORD.length})` : '✗ NOT SET IN ENV');
        console.log('  Match:', password === process.env.ADMIN_PASSWORD ? '✓ YES' : '✗ NO');

        // Validate admin password
        if (!password) {
          console.log('❌ Validation failed: No password provided');
          return res.status(400).json({
            success: false,
            message: 'Authentication required'
          });
        }

        if (password !== process.env.ADMIN_PASSWORD) {
          console.log('❌ Validation failed: Password mismatch');
          return res.status(404).json({
            success: false,
            message: 'Not found'
          });
        }

        console.log('✓ Password validated');

        // Validate required blog data
        console.log('Field validation:');
        console.log('  Title:', title ? '✓' : '✗');
        console.log('  Excerpt:', excerpt ? '✓' : '✗');
        console.log('  Link:', link ? '✓' : '✗');

        if (!title || !excerpt || !link) {
          console.log('❌ Validation failed: Missing required fields');
          return res.status(400).json({
            success: false,
            message: 'Title, excerpt, and link are required'
          });
        }

        console.log('✓ All validations passed');
        console.log('📧 Sending emails...');

        // Use EmailService
        const result = await EmailService.sendBlogNotification({
          title,
          excerpt,
          link,
          imageUrl,
          author,
          publishedDate,
          category,
          tags: Array.isArray(tags) ? tags : []
        });

        console.log(`✅ Success: Sent to ${result.sent} subscribers`);
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');

        return res.status(200).json({
          success: true,
          message: `Blog notification sent to ${result.sent} subscribers`,
          details: result
        });
      } catch (error) {
        console.error('❌ Error in notify route:', error.message);
        console.error('Stack:', error.stack);
        console.error('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');

        return res.status(500).json({
          success: false,
          message: 'Failed to send notifications',
          error: ExpressService.nodeEnvironment === 'development' ? error.message : undefined
        });
      }
    });
  }

  /**
   * POST /api/subscriber/all
   * Get all subscribers (ADMIN ONLY)
   */
  #allSubscribersRoute() {
    this.#subscriberRouter.post('/all', async (req, res, next) => {
      console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
      console.log('🔵 POST /api/subscriber/all');

      try {
        const { password } = req.body;

        console.log('Password check:');
        console.log('  Received:', password ? `✓ (length: ${password.length})` : '✗ MISSING');
        console.log('  Expected:', process.env.ADMIN_PASSWORD ? `✓ (length: ${process.env.ADMIN_PASSWORD.length})` : '✗ NOT SET');
        console.log('  Match:', password === process.env.ADMIN_PASSWORD ? '✓ YES' : '✗ NO');

        if (!password || password !== process.env.ADMIN_PASSWORD) {
          console.log('❌ Authentication failed');
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');
          return res.status(404).json({
            success: false,
            message: 'Not found'
          });
        }

        console.log('✓ Password validated, fetching all subscribers...');

        const subscribers = await SubscriptionService.getAllSubscribers();

        console.log(`✅ Retrieved ${subscribers.length} subscribers`);
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');

        return res.status(200).json({
          success: true,
          total: subscribers.length,
          subscribers: subscribers
        });
      } catch (error) {
        console.error('❌ Error in all subscribers route:', error);
        console.error('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');

        return res.status(500).json({
          success: false,
          message: 'Failed to retrieve subscribers',
          error: ExpressService.nodeEnvironment === 'development' ? error.message : undefined
        });
      }
    });
  }

  /**
   * POST /api/subscriber/stats
   * Get subscriber statistics (ADMIN ONLY)
   */
  #statsRoute() {
    this.#subscriberRouter.post('/stats', async (req, res, next) => {
      console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
      console.log('🔵 POST /api/subscriber/stats');

      try {
        const { password } = req.body;

        console.log('Password check:');
        console.log('  Received:', password ? `✓ (length: ${password.length})` : '✗ MISSING');
        console.log('  Expected:', process.env.ADMIN_PASSWORD ? `✓ (length: ${process.env.ADMIN_PASSWORD.length})` : '✗ NOT SET');
        console.log('  Match:', password === process.env.ADMIN_PASSWORD ? '✓ YES' : '✗ NO');

        if (!password || password !== process.env.ADMIN_PASSWORD) {
          console.log('❌ Authentication failed');
          console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');
          return res.status(404).json({
            success: false,
            message: 'Not found'
          });
        }

        console.log('✓ Password validated, fetching stats...');

        const stats = await SubscriptionService.getStats();
        const subscribers = await SubscriptionService.getAllSubscribers();

        console.log('✅ Stats retrieved successfully');
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');

        return res.status(200).json({
          success: true,
          stats,
          recentSubscribers: subscribers.slice(-5).reverse()
        });
      } catch (error) {
        console.error('❌ Error in stats route:', error);
        console.error('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');

        return res.status(500).json({
          success: false,
          message: 'Failed to retrieve stats',
          error: ExpressService.nodeEnvironment === 'development' ? error.message : undefined
        });
      }
    });
  }
  /**
     * GET /api/subscriber/articles
     * Get all articles with optional search
     */
  #articlesRoute() {
    this.#subscriberRouter.get('/articles', async (req, res, next) => {
      try {
        const { search } = req.query;
        const articles = search
          ? ArticleService.searchArticles(search)
          : ArticleService.getAllArticles();

        return res.status(200).json({
          success: true,
          articles: articles.map(a => ({
            slug: a.slug,
            title: a.title,
            excerpt: a.excerpt,
            category: a.category,
            publish_date: a.publish_date,
            featured_image_url: a.featured_image_url,
            author: a.author,
            status: a.status
          }))
        });
      } catch (error) {
        console.error('❌ Error fetching articles:', error);
        return res.status(500).json({
          success: false,
          message: 'Failed to fetch articles'
        });
      }
    });
  }

  /**
   * GET /api/subscriber/articles/:slug
   * Get specific article data for email
   */
  #articleDetailRoute() {
    this.#subscriberRouter.get('/articles/:slug', async (req, res, next) => {
      try {
        const { slug } = req.params;
        const articleData = ArticleService.getArticleForEmail(slug);

        if (!articleData) {
          return res.status(404).json({
            success: false,
            message: 'Article not found'
          });
        }

        return res.status(200).json({
          success: true,
          article: articleData
        });
      } catch (error) {
        console.error('❌ Error fetching article:', error);
        return res.status(500).json({
          success: false,
          message: 'Failed to fetch article'
        });
      }
    });
  }


  /**
   * Health check route
   */
  #healthCheckRoute() {
    // Mount health check on both paths
    this.#app.get('/backend/api/health', (req, res) => {
      res.status(200).json({
        status: 'ok',
        emailConnected: EmailService.isConnected,
        timestamp: new Date().toISOString(),
        environment: ExpressService.nodeEnvironment
      });
    });

    this.#app.get('/api/health', (req, res) => {
      res.status(200).json({
        status: 'ok',
        emailConnected: EmailService.isConnected,
        timestamp: new Date().toISOString(),
        environment: ExpressService.nodeEnvironment
      });
    });
  }

  /**
   * Error handling middleware
   */
  #errorHandler() {
    this.#app.use((err, req, res, next) => {
      console.error('❌ Unhandled error:', err);
      res.status(500).json({
        success: false,
        message: 'Internal server error',
        error: ExpressService.nodeEnvironment === 'development' ? err.message : undefined
      });
    });
  }

  /**
 * Start the Express server with comprehensive error handling
 */
  async start() {
    try {
      console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
      console.log('🚀 Starting TLC Homes Newsletter Backend...');
      console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');

      // Log environment info
      console.log('📊 Environment Check:');
      console.log(`   NODE_ENV: ${ExpressService.nodeEnvironment}`);
      console.log(`   PORT: ${ExpressService.port}`);
      console.log(`   EMAIL_USER set: ${process.env.EMAIL_USER ? '✓' : '✗'}`);
      console.log(`   EMAIL_PASSWORD set: ${process.env.EMAIL_PASSWORD ? '✓' : '✗'}`);
      console.log(`   ADMIN_PASSWORD set: ${process.env.ADMIN_PASSWORD ? '✓' : '✗'}`);
      console.log(`   Working directory: ${process.cwd()}`);
      console.log(`   Script location: ${ExpressService.dirname}\n`);

      // Initialize email service with better error handling
      console.log('📧 Initializing email service...');
      let emailConnected = false;

      try {
        emailConnected = await EmailService.initialize();
        console.log(`   Email service: ${emailConnected ? '✓ Connected' : '✗ Not Connected'}\n`);
      } catch (emailError) {
        console.error('   ⚠️  Email service initialization failed:');
        console.error('   Error:', emailError.message);
        console.error('   Stack:', emailError.stack);
        console.log('   Continuing without email service...\n');
      }

      if (!emailConnected) {
        console.log('⚠️  WARNING: Email service not available');
        console.log('   - Subscribers can sign up, but emails will NOT be sent');
        console.log('   - Check EMAIL_USER/EMAIL_PASSWORD in environment variables\n');
      }

      // Initialize Google Sheets service
      console.log('📊 Initializing Google Sheets service...');
      let sheetsConnected = false;

      try {
        sheetsConnected = await GoogleSheetsService.initialize();
        console.log(`   Google Sheets: ${sheetsConnected ? '✓ Connected' : '✗ Not Connected'}\n`);
      } catch (sheetsError) {
        console.error('   ⚠️  Google Sheets initialization failed:');
        console.error('   Error:', sheetsError.message);
        console.log('   Continuing without Google Sheets integration...\n');
      }

      if (!sheetsConnected) {
        console.log('⚠️  WARNING: Google Sheets integration not available');
        console.log('   - Subscribers will still be saved locally');
        console.log('   - Set GOOGLE_APPS_SCRIPT_URL in environment to enable\n');
      }

      // Initialize middleware
      console.log('⚙️  Initializing middleware...');
      this.#initialize();
      console.log('   ✓ Middleware initialized\n');

      // Add all routes
      console.log('🛣️  Registering routes...');
      this.#healthCheckRoute();
      this.#addSubscriberRoute();
      this.#verifySubscriberRoute();
      this.#unsubscribeRoute();
      this.#notifySubscribersRoute();
      this.#statsRoute();
      this.#allSubscribersRoute();
      this.#articlesRoute();
      this.#articleDetailRoute();
      console.log('   ✓ All routes registered\n');

      // Error handler (must be last)
      this.#errorHandler();

      // Start listening
      console.log('🎧 Starting HTTP server...');
      this.#app.listen(ExpressService.port, () => {
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        console.log(`✅ TLC Homes Newsletter Backend STARTED`);
        console.log(`📡 Server: http://localhost:${ExpressService.port}`);
        console.log(`🌍 Environment: ${ExpressService.nodeEnvironment}`);
        console.log(`📧 Email: ${emailConnected ? '✓ Connected' : '✗ Not Connected'}`);
        console.log(`⏰ Started at: ${new Date().toISOString()}`);
        console.log('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');

        if (!emailConnected) {
          console.log('⚠️  Backend running but emails disabled!\n');
        }
      });

      // Handle server errors
      this.#app.on('error', (error) => {
        console.error('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
        console.error('❌ HTTP SERVER ERROR:');
        console.error('Error code:', error.code);
        console.error('Error message:', error.message);
        console.error('Error stack:', error.stack);
        console.error('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');

        if (error.code === 'EADDRINUSE') {
          console.error(`Port ${ExpressService.port} is already in use!`);
          console.error('Try a different port or kill the existing process.\n');
        }
      });

    } catch (error) {
      console.error('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━');
      console.error('❌ CRITICAL: Failed to start server');
      console.error('Error name:', error.name);
      console.error('Error message:', error.message);
      console.error('Error stack:', error.stack);
      console.error('━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━━\n');

      // Don't exit immediately - log more info
      console.error('Environment variables:');
      console.error('  NODE_ENV:', process.env.NODE_ENV);
      console.error('  PORT:', process.env.PORT);
      console.error('  EMAIL_USER:', process.env.EMAIL_USER ? '[SET]' : '[NOT SET]');
      console.error('  EMAIL_PASSWORD:', process.env.EMAIL_PASSWORD ? '[SET]' : '[NOT SET]');
      console.error('\n');

      process.exit(1);
    }
  }
}

export default ExpressService;